'use client';

import React, { useState } from 'react';
import { useParams, useRouter } from 'next/navigation';
import {
    Box,
    Container,
    Typography,
    Grid,
    Card,
    CardContent,
    Avatar,
    Rating,
    Button,
    Chip,
    Tabs,
    Tab,
    Divider,
    List,
    ListItem,
    ListItemIcon,
    ListItemText,
    Accordion,
    AccordionSummary,
    AccordionDetails,
    Paper,
    IconButton,
} from '@mui/material';
import {
    CheckCircle,
    Star,
    AccessTime,
    Refresh,
    ExpandMore,
    ArrowBack,
    ArrowForward,
    Favorite,
    FavoriteBorder,
    Share,
} from '@mui/icons-material';
import Header from '@/components/Header';
import Footer from '@/components/Footer';
import ServiceCard from '@/components/ServiceCard';
import { services, freelancers, reviews } from '@/data/mockData';
import { serviceDetails } from '@/data/serviceDetails';

export default function ServiceDetailPage() {
    const params = useParams();
    const router = useRouter();
    const serviceId = parseInt(params.id);

    const service = services.find((s) => s.id === serviceId);
    const details = serviceDetails[serviceId];
    const serviceReviews = reviews.filter((r) => r.serviceId === serviceId);
    const relatedServices = services.filter((s) => s.category === service?.category && s.id !== serviceId).slice(0, 3);

    const [selectedTab, setSelectedTab] = useState(0);
    const [selectedPackage, setSelectedPackage] = useState(1);
    const [currentImageIndex, setCurrentImageIndex] = useState(0);
    const [isFavorite, setIsFavorite] = useState(false);

    if (!service) {
        return (
            <>
                <Header />
                <Container sx={{ py: 8, textAlign: 'center' }}>
                    <Typography variant="h4">Service not found</Typography>
                    <Button variant="contained" onClick={() => router.push('/services')} sx={{ mt: 2 }}>
                        Browse Services
                    </Button>
                </Container>
                <Footer />
            </>
        );
    }

    const freelancer = freelancers.find((f) => f.id === service.provider.id);
    const gallery = details?.gallery || [service.image];

    const handleOrder = () => {
        router.push(`/checkout?service=${serviceId}&package=${selectedPackage}`);
    };

    const currentPackage = details?.packages?.[selectedPackage];

    return (
        <>
            <Header />

            <Container maxWidth="lg" sx={{ mt: 4, mb: 8 }}>
                {/* Breadcrumb */}
                <Box sx={{ mb: 3 }}>
                    <Button startIcon={<ArrowBack />} onClick={() => router.push('/services')}>
                        Back to Services
                    </Button>
                </Box>

                <Grid container spacing={4}>
                    {/* Left Column - Service Details */}
                    <Grid item xs={12} md={8}>
                        {/* Image Gallery */}
                        <Card sx={{ mb: 3 }}>
                            <Box sx={{ position: 'relative' }}>
                                <Box
                                    component="img"
                                    src={gallery[currentImageIndex]}
                                    alt={service.title}
                                    sx={{
                                        width: '100%',
                                        height: 400,
                                        objectFit: 'cover',
                                    }}
                                />
                                {gallery.length > 1 && (
                                    <>
                                        <IconButton
                                            sx={{
                                                position: 'absolute',
                                                left: 10,
                                                top: '50%',
                                                transform: 'translateY(-50%)',
                                                bgcolor: 'rgba(255,255,255,0.9)',
                                                '&:hover': { bgcolor: 'white' },
                                            }}
                                            onClick={() => setCurrentImageIndex((prev) => (prev - 1 + gallery.length) % gallery.length)}
                                        >
                                            <ArrowBack />
                                        </IconButton>
                                        <IconButton
                                            sx={{
                                                position: 'absolute',
                                                right: 10,
                                                top: '50%',
                                                transform: 'translateY(-50%)',
                                                bgcolor: 'rgba(255,255,255,0.9)',
                                                '&:hover': { bgcolor: 'white' },
                                            }}
                                            onClick={() => setCurrentImageIndex((prev) => (prev + 1) % gallery.length)}
                                        >
                                            <ArrowForward />
                                        </IconButton>
                                    </>
                                )}
                            </Box>
                            {gallery.length > 1 && (
                                <Box sx={{ display: 'flex', gap: 1, p: 2, overflowX: 'auto' }}>
                                    {gallery.map((img, index) => (
                                        <Box
                                            key={index}
                                            component="img"
                                            src={img}
                                            alt={`Gallery ${index + 1}`}
                                            onClick={() => setCurrentImageIndex(index)}
                                            sx={{
                                                width: 80,
                                                height: 60,
                                                objectFit: 'cover',
                                                borderRadius: 1,
                                                cursor: 'pointer',
                                                border: currentImageIndex === index ? '2px solid' : 'none',
                                                borderColor: 'primary.main',
                                                opacity: currentImageIndex === index ? 1 : 0.6,
                                                '&:hover': { opacity: 1 },
                                            }}
                                        />
                                    ))}
                                </Box>
                            )}
                        </Card>

                        {/* Title and Actions */}
                        <Box sx={{ mb: 3 }}>
                            <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'flex-start', mb: 2 }}>
                                <Typography variant="h4" fontWeight={700}>
                                    {service.title}
                                </Typography>
                                <Box sx={{ display: 'flex', gap: 1 }}>
                                    <IconButton onClick={() => setIsFavorite(!isFavorite)}>
                                        {isFavorite ? <Favorite color="error" /> : <FavoriteBorder />}
                                    </IconButton>
                                    <IconButton>
                                        <Share />
                                    </IconButton>
                                </Box>
                            </Box>

                            <Box sx={{ display: 'flex', alignItems: 'center', gap: 2, flexWrap: 'wrap' }}>
                                <Chip label={service.category} color="primary" />
                                <Box sx={{ display: 'flex', alignItems: 'center', gap: 0.5 }}>
                                    <Rating value={service.rating} precision={0.1} readOnly size="small" />
                                    <Typography variant="body2" fontWeight={600}>
                                        {service.rating}
                                    </Typography>
                                    <Typography variant="body2" color="text.secondary">
                                        ({service.reviewCount} reviews)
                                    </Typography>
                                </Box>
                            </Box>
                        </Box>

                        {/* Tabs */}
                        <Card>
                            <Tabs value={selectedTab} onChange={(e, v) => setSelectedTab(v)}>
                                <Tab label="Overview" />
                                <Tab label={`Reviews (${serviceReviews.length})`} />
                                {details?.faqs && <Tab label="FAQ" />}
                            </Tabs>
                            <Divider />

                            <CardContent>
                                {/* Overview Tab */}
                                {selectedTab === 0 && (
                                    <Box>
                                        <Typography variant="h6" fontWeight={600} gutterBottom>
                                            About This Service
                                        </Typography>
                                        <Typography
                                            variant="body1"
                                            sx={{ whiteSpace: 'pre-line', lineHeight: 1.8 }}
                                        >
                                            {details?.description || 'Professional service delivered with quality and care.'}
                                        </Typography>
                                    </Box>
                                )}

                                {/* Reviews Tab */}
                                {selectedTab === 1 && (
                                    <Box>
                                        {serviceReviews.length === 0 ? (
                                            <Typography color="text.secondary">No reviews yet</Typography>
                                        ) : (
                                            <List>
                                                {serviceReviews.map((review) => (
                                                    <React.Fragment key={review.id}>
                                                        <ListItem alignItems="flex-start" sx={{ px: 0 }}>
                                                            <Box sx={{ width: '100%' }}>
                                                                <Box sx={{ display: 'flex', gap: 2, mb: 1 }}>
                                                                    <Avatar src={review.clientAvatar} alt={review.clientName} />
                                                                    <Box sx={{ flex: 1 }}>
                                                                        <Typography variant="subtitle2" fontWeight={600}>
                                                                            {review.clientName}
                                                                        </Typography>
                                                                        <Box sx={{ display: 'flex', alignItems: 'center', gap: 1 }}>
                                                                            <Rating value={review.rating} size="small" readOnly />
                                                                            <Typography variant="caption" color="text.secondary">
                                                                                {review.date}
                                                                            </Typography>
                                                                        </Box>
                                                                    </Box>
                                                                </Box>
                                                                <Typography variant="body2" sx={{ ml: 7 }}>
                                                                    {review.comment}
                                                                </Typography>
                                                            </Box>
                                                        </ListItem>
                                                        <Divider sx={{ my: 2 }} />
                                                    </React.Fragment>
                                                ))}
                                            </List>
                                        )}
                                    </Box>
                                )}

                                {/* FAQ Tab */}
                                {selectedTab === 2 && details?.faqs && (
                                    <Box>
                                        {details.faqs.map((faq, index) => (
                                            <Accordion key={index}>
                                                <AccordionSummary expandIcon={<ExpandMore />}>
                                                    <Typography fontWeight={600}>{faq.question}</Typography>
                                                </AccordionSummary>
                                                <AccordionDetails>
                                                    <Typography>{faq.answer}</Typography>
                                                </AccordionDetails>
                                            </Accordion>
                                        ))}
                                    </Box>
                                )}
                            </CardContent>
                        </Card>

                        {/* Related Services */}
                        {relatedServices.length > 0 && (
                            <Box sx={{ mt: 4 }}>
                                <Typography variant="h5" fontWeight={700} gutterBottom>
                                    Related Services
                                </Typography>
                                <Grid container spacing={3}>
                                    {relatedServices.map((relatedService) => (
                                        <Grid item xs={12} sm={4} key={relatedService.id}>
                                            <ServiceCard {...relatedService} />
                                        </Grid>
                                    ))}
                                </Grid>
                            </Box>
                        )}
                    </Grid>

                    {/* Right Column - Pricing & Seller Info */}
                    <Grid item xs={12} md={4}>
                        {/* Pricing Packages */}
                        {details?.packages && (
                            <Card sx={{ mb: 3, position: 'sticky', top: 20 }}>
                                <CardContent>
                                    <Tabs
                                        value={selectedPackage}
                                        onChange={(e, v) => setSelectedPackage(v)}
                                        variant="fullWidth"
                                    >
                                        {details.packages.map((pkg, index) => (
                                            <Tab
                                                key={index}
                                                label={
                                                    <Box>
                                                        {pkg.name}
                                                        {pkg.popular && (
                                                            <Chip label="Popular" size="small" color="primary" sx={{ ml: 1 }} />
                                                        )}
                                                    </Box>
                                                }
                                            />
                                        ))}
                                    </Tabs>

                                    {currentPackage && (
                                        <Box sx={{ mt: 3 }}>
                                            <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', mb: 2 }}>
                                                <Typography variant="h4" fontWeight={700} color="primary.main">
                                                    ${currentPackage.price}
                                                </Typography>
                                                <Box sx={{ display: 'flex', alignItems: 'center', gap: 0.5 }}>
                                                    <AccessTime fontSize="small" />
                                                    <Typography variant="body2">{currentPackage.deliveryTime}</Typography>
                                                </Box>
                                            </Box>

                                            <List dense>
                                                {currentPackage.features.map((feature, index) => (
                                                    <ListItem key={index} sx={{ px: 0 }}>
                                                        <ListItemIcon sx={{ minWidth: 32 }}>
                                                            <CheckCircle color="success" fontSize="small" />
                                                        </ListItemIcon>
                                                        <ListItemText primary={feature} />
                                                    </ListItem>
                                                ))}
                                            </List>

                                            <Button
                                                variant="contained"
                                                size="large"
                                                fullWidth
                                                onClick={handleOrder}
                                                sx={{ mt: 2 }}
                                            >
                                                Order Now (${currentPackage.price})
                                            </Button>
                                        </Box>
                                    )}
                                </CardContent>
                            </Card>
                        )}

                        {/* Seller Info */}
                        {freelancer && (
                            <Card>
                                <CardContent>
                                    <Typography variant="h6" fontWeight={600} gutterBottom>
                                        About the Seller
                                    </Typography>

                                    <Box
                                        sx={{
                                            display: 'flex',
                                            alignItems: 'center',
                                            gap: 2,
                                            mb: 2,
                                            cursor: 'pointer',
                                        }}
                                        onClick={() => router.push(`/freelancers/${freelancer.id}`)}
                                    >
                                        <Avatar
                                            src={freelancer.avatar}
                                            alt={freelancer.name}
                                            sx={{ width: 60, height: 60 }}
                                        />
                                        <Box>
                                            <Typography variant="subtitle1" fontWeight={600}>
                                                {freelancer.name}
                                            </Typography>
                                            <Typography variant="body2" color="text.secondary">
                                                {freelancer.title}
                                            </Typography>
                                            <Box sx={{ display: 'flex', alignItems: 'center', gap: 0.5, mt: 0.5 }}>
                                                <Star fontSize="small" sx={{ color: '#f59e0b' }} />
                                                <Typography variant="body2" fontWeight={600}>
                                                    {freelancer.rating}
                                                </Typography>
                                                <Typography variant="body2" color="text.secondary">
                                                    ({freelancer.reviews})
                                                </Typography>
                                            </Box>
                                        </Box>
                                    </Box>

                                    <Divider sx={{ my: 2 }} />

                                    <Box sx={{ display: 'flex', justifyContent: 'space-between', mb: 1 }}>
                                        <Typography variant="body2" color="text.secondary">
                                            Location:
                                        </Typography>
                                        <Typography variant="body2" fontWeight={600}>
                                            {freelancer.location}
                                        </Typography>
                                    </Box>

                                    <Box sx={{ display: 'flex', justifyContent: 'space-between', mb: 1 }}>
                                        <Typography variant="body2" color="text.secondary">
                                            Member since:
                                        </Typography>
                                        <Typography variant="body2" fontWeight={600}>
                                            {freelancer.memberSince}
                                        </Typography>
                                    </Box>

                                    <Box sx={{ display: 'flex', justifyContent: 'space-between', mb: 1 }}>
                                        <Typography variant="body2" color="text.secondary">
                                            Response time:
                                        </Typography>
                                        <Typography variant="body2" fontWeight={600}>
                                            {freelancer.responseTime}
                                        </Typography>
                                    </Box>

                                    <Button
                                        variant="outlined"
                                        fullWidth
                                        sx={{ mt: 2 }}
                                        onClick={() => router.push(`/freelancers/${freelancer.id}`)}
                                    >
                                        View Profile
                                    </Button>
                                </CardContent>
                            </Card>
                        )}
                    </Grid>
                </Grid>
            </Container>

            <Footer />
        </>
    );
}
